﻿//***************************************************************************************
// TreeSprite.fx Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// Wykorzystuje shader geometrii, aby rozszerzyć "sprite'y" punktowe do wyrównanych względem osi y 
// i zwróconych przodem do kamery obiektów typu billboard.
//***************************************************************************************

#include "LightHelper.fx"
 
cbuffer cbPerFrame
{
	DirectionalLight gDirLights[3];
	float3 gEyePosW;

	float  gFogStart;
	float  gFogRange;
	float4 gFogColor;
};

cbuffer cbPerObject
{
	float4x4 gViewProj;
	Material gMaterial;
};

cbuffer cbFixed
{
	//
	// Oblicz współrzędne tekstury do jej rozciągnięcia na czworokącie.
	//

	float2 gTexC[4] = 
	{
		float2(0.0f, 1.0f),
		float2(0.0f, 0.0f),
		float2(1.0f, 1.0f),
		float2(1.0f, 0.0f)
	};
};

// Danych nienumerycznych nie można zapisywać w obiekcie cbuffer.
Texture2DArray gTreeMapArray;

SamplerState samLinear
{
	Filter   = MIN_MAG_MIP_LINEAR;
	AddressU = CLAMP;
	AddressV = CLAMP;
};

struct VertexIn
{
	float3 PosW  : POSITION;
	float2 SizeW : SIZE;
};

struct VertexOut
{
	float3 CenterW : POSITION;
	float2 SizeW   : SIZE;
};

struct GeoOut
{
	float4 PosH    : SV_POSITION;
    float3 PosW    : POSITION;
    float3 NormalW : NORMAL;
    float2 Tex     : TEXCOORD;
    uint   PrimID  : SV_PrimitiveID;
};

VertexOut VS(VertexIn vin)
{
	VertexOut vout;

	// Przekaż dane bezpośrednio do shadera geometrii.
	vout.CenterW = vin.PosW;
	vout.SizeW   = vin.SizeW;

	return vout;
}
 
// Rozszerzamy każdy punkt do czworokąta (4 wierzchołków), a więc maksymalna liczba
// wierzchołków zwracanych przez wywołanie shadera geometrii wynosi 4.
[maxvertexcount(4)]
void GS(point VertexOut gin[1], 
        uint primID : SV_PrimitiveID, 
        inout TriangleStream<GeoOut> triStream)
{	
	//
    // Oblicz lokalny układ współrzędnych â€žspriteâ€™uâ€ť względem
    // przestrzeni świata tak, aby wyrównać billboard względem osi y 
    // i odwrócić go w stronę oka obserwatora.
	//

	float3 up = float3(0.0f, 1.0f, 0.0f);
	float3 look = gEyePosW - gin[0].CenterW;
	look.y = 0.0f; // y-axis aligned, so project to xz-plane
	look = normalize(look);
	float3 right = cross(up, look);

	//
	// Oblicz wierzchołki połączenia trójkątów (czworokąta) w przestrzeni świata.
	//
	float halfWidth  = 0.5f*gin[0].SizeW.x;
	float halfHeight = 0.5f*gin[0].SizeW.y;
	
	float4 v[4];
	v[0] = float4(gin[0].CenterW + halfWidth*right - halfHeight*up, 1.0f);
	v[1] = float4(gin[0].CenterW + halfWidth*right + halfHeight*up, 1.0f);
	v[2] = float4(gin[0].CenterW - halfWidth*right - halfHeight*up, 1.0f);
	v[3] = float4(gin[0].CenterW - halfWidth*right + halfHeight*up, 1.0f);

	//
    // Przekształć wierzchołki czworokąta do przestrzeni świata i zwróć 
    // je jako połączenie trójkątów.
	//
	GeoOut gout;
	[unroll]
	for(int i = 0; i < 4; ++i)
	{
		gout.PosH     = mul(v[i], gViewProj);
		gout.PosW     = v[i].xyz;
		gout.NormalW  = look;
		gout.Tex      = gTexC[i];
		gout.PrimID   = primID;
		
		triStream.Append(gout);
	}
}

float4 PS(GeoOut pin, uniform int gLightCount, uniform bool gUseTexure, uniform bool gAlphaClip, uniform bool gFogEnabled) : SV_Target
{
	// Normalizuj ponownie normalną (mogła ulec denormalizacji w procesie interpolacji).
    pin.NormalW = normalize(pin.NormalW);

	// Wektor toEye jest używany przy oświetlaniu.
	float3 toEye = gEyePosW - pin.PosW;

	// Przechowaj odległość tego punktu powierzchni od oka.
	float distToEye = length(toEye);

	// Normalizuj.
	toEye /= distToEye;
   
    // Wartość domyślna â€“ element neutralny mnożenia.
    float4 texColor = float4(1, 1, 1, 1);
    if(gUseTexure)
	{
		// Próbkuj teksturę.
		float3 uvw = float3(pin.Tex, pin.PrimID%4);
		texColor = gTreeMapArray.Sample( samLinear, uvw );

		if(gAlphaClip)
		{
         	// Odrzuć piksel, jeśli alfa tekstury < 0,05. Zauważ, że 
         	// ten test wykonujemy tak szybko, jak to możliwe, aby
         	// jak najszybciej opuścić shader, pomijając tym samym
         	// pozostałą część jego kodu.
			clip(texColor.a - 0.05f);
		}
	}

	//
	// Oświetlenie.
	//

	float4 litColor = texColor;
	if( gLightCount > 0  )
	{
		// Rozpocznij sumowanie od zera.
		float4 ambient = float4(0.0f, 0.0f, 0.0f, 0.0f);
		float4 diffuse = float4(0.0f, 0.0f, 0.0f, 0.0f);
		float4 spec    = float4(0.0f, 0.0f, 0.0f, 0.0f);

		// Sumuj udział światła z każdego źródła. 
		[unroll]
		for(int i = 0; i < gLightCount; ++i)
		{
			float4 A, D, S;
			ComputeDirectionalLight(gMaterial, gDirLights[i], pin.NormalW, toEye, 
				A, D, S);

			ambient += A;
			diffuse += D;
			spec    += S;
		}

		// Modulacja z opóźnionym dodaniem.
		litColor = texColor*(ambient + diffuse) + spec;
	}

	//
	// Mgła
	//

	if( gFogEnabled )
	{
		float fogLerp = saturate( (distToEye - gFogStart) / gFogRange ); 

		// Zmieszaj kolor mgły i oświetlony kolor.
		litColor = lerp(litColor, gFogColor, fogLerp);
	}

	// Pobierz wartość alfa z koloru materiału rozpraszającego i tekstury.
	litColor.a = gMaterial.Diffuse.a * texColor.a;

    return litColor;
}

//---------------------------------------------------------------------------------------
// Zdefiniowano tylko techniki wykorzystywane przez naszą aplikację; w razie potrzeby 
// można zdefiniować inne warianty.
//---------------------------------------------------------------------------------------
technique11 Light3
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( CompileShader( gs_5_0, GS() ) );
        SetPixelShader( CompileShader( ps_5_0, PS(3, false, false, false) ) );
    }
}

technique11 Light3TexAlphaClip
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( CompileShader( gs_5_0, GS() ) );
        SetPixelShader( CompileShader( ps_5_0, PS(3, true, true, false) ) );
    }
}
            
technique11 Light3TexAlphaClipFog
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( CompileShader( gs_5_0, GS() ) );
        SetPixelShader( CompileShader( ps_5_0, PS(3, true, true, true) ) );
    }
}
